/**
 * @license
 * Copyright 2025 Google LLC
 * SPDX-License-Identifier: Apache-2.0
 */
import { useState, useCallback, useEffect } from 'react';
import { FolderTrustChoice } from '../components/FolderTrustDialog.js';
import { loadTrustedFolders, TrustLevel, isWorkspaceTrusted, } from '../../config/trustedFolders.js';
import * as process from 'node:process';
export const useFolderTrust = (settings, onTrustChange) => {
    const [isTrusted, setIsTrusted] = useState(undefined);
    const [isFolderTrustDialogOpen, setIsFolderTrustDialogOpen] = useState(false);
    const [isRestarting, setIsRestarting] = useState(false);
    const folderTrust = settings.merged.security?.folderTrust?.enabled;
    useEffect(() => {
        const { isTrusted: trusted } = isWorkspaceTrusted(settings.merged);
        setIsTrusted(trusted);
        setIsFolderTrustDialogOpen(trusted === undefined);
        onTrustChange(trusted);
    }, [folderTrust, onTrustChange, settings.merged]);
    const handleFolderTrustSelect = useCallback((choice) => {
        const trustedFolders = loadTrustedFolders();
        const cwd = process.cwd();
        let trustLevel;
        const wasTrusted = isTrusted ?? true;
        switch (choice) {
            case FolderTrustChoice.TRUST_FOLDER:
                trustLevel = TrustLevel.TRUST_FOLDER;
                break;
            case FolderTrustChoice.TRUST_PARENT:
                trustLevel = TrustLevel.TRUST_PARENT;
                break;
            case FolderTrustChoice.DO_NOT_TRUST:
                trustLevel = TrustLevel.DO_NOT_TRUST;
                break;
            default:
                return;
        }
        trustedFolders.setValue(cwd, trustLevel);
        const currentIsTrusted = trustLevel === TrustLevel.TRUST_FOLDER ||
            trustLevel === TrustLevel.TRUST_PARENT;
        setIsTrusted(currentIsTrusted);
        onTrustChange(currentIsTrusted);
        const needsRestart = wasTrusted !== currentIsTrusted;
        if (needsRestart) {
            setIsRestarting(true);
            setIsFolderTrustDialogOpen(true);
        }
        else {
            setIsFolderTrustDialogOpen(false);
        }
    }, [onTrustChange, isTrusted]);
    return {
        isTrusted,
        isFolderTrustDialogOpen,
        handleFolderTrustSelect,
        isRestarting,
    };
};
//# sourceMappingURL=useFolderTrust.js.map