"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerGettingStartedRoutes = registerGettingStartedRoutes;
var _configSchema = require("@osd/config-schema");
var _stream = require("stream");
var _utils = require("../../../../../src/core/server/saved_objects/routes/utils");
var _helper = require("./helper");
var _utils2 = require("../../../../../src/core/server/utils");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerGettingStartedRoutes(router) {
  // Fetch the tutorial assets
  router.get({
    path: `/api/observability/gettingStarted/{tutorialId}`,
    validate: {
      params: _configSchema.schema.object({
        tutorialId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const fileData = await (0, _helper.loadAssetsFromFile)(request.params.tutorialId);
      return response.ok({
        body: {
          data: fileData
        }
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: 'Issue in fetching NDJSON file for tutorialId: ' + request.params.tutorialId
      });
    }
  });

  // Fetch the tutorial saved searches
  router.get({
    path: `/api/observability/gettingStarted/indexPatterns/{tutorialId}`,
    validate: {
      params: _configSchema.schema.object({
        tutorialId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    try {
      const fileData = await (0, _helper.loadAssetsFromFile)(request.params.tutorialId);
      const objects = await (0, _utils.createSavedObjectsStreamFromNdJson)(_stream.Readable.from(fileData));
      const loadedObjects = await objects.toArray();
      const loadDashboardIds = loadedObjects.filter(savedObject => savedObject.type === 'index-pattern').map(indexPattern => ({
        id: indexPattern.id,
        title: indexPattern.attributes.title
      }));
      return response.ok({
        body: {
          data: loadDashboardIds
        }
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: 'Issue in fetching index-patterns for tutorialId: ' + request.params.tutorialId
      });
    }
  });
  router.post({
    path: `/api/observability/gettingStarted/createAssets`,
    validate: {
      body: _configSchema.schema.object({
        mdsId: _configSchema.schema.string(),
        mdsLabel: _configSchema.schema.string(),
        tutorialId: _configSchema.schema.string(),
        indexTemplates: _configSchema.schema.arrayOf(_configSchema.schema.object({
          name: _configSchema.schema.string(),
          templatePath: _configSchema.schema.string(),
          template: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any())
        }), {
          defaultValue: []
        })
      })
    }
  }, async (context, request, response) => {
    try {
      const {
        mdsId,
        mdsLabel,
        tutorialId,
        indexTemplates
      } = request.body;
      const {
        requestWorkspaceId
      } = (0, _utils2.getWorkspaceState)(request);
      const fileData = await (0, _helper.loadAssetsFromFile)(tutorialId);

      // create related index templates
      if (indexTemplates.length > 0) {
        await (0, _helper.createAllTemplatesSettled)(context, indexTemplates, mdsId);
      }
      const objects = await (0, _utils.createSavedObjectsStreamFromNdJson)(_stream.Readable.from(fileData));
      const loadedObjects = await objects.toArray();
      const updatedObjects = loadedObjects.map(obj => {
        let newId = obj.id;
        let references = obj.references;
        if (requestWorkspaceId) {
          var _references;
          newId = `workspaceId-${requestWorkspaceId}-${newId}`;
          references = (_references = references) === null || _references === void 0 ? void 0 : _references.map(ref => {
            return {
              ...ref,
              id: `workspaceId-${requestWorkspaceId}-${ref.id}`
            };
          });
        }
        if (mdsId) {
          var _references2;
          newId = `mds-${mdsId}-objectId-${obj.id}`;
          references = (_references2 = references) === null || _references2 === void 0 ? void 0 : _references2.map(ref => {
            return {
              ...ref,
              id: `mds-${mdsId}-objectId-${ref.id}`
            };
          });
          if (obj.type === 'index-pattern' && references) {
            references.push({
              id: mdsId,
              type: 'data-source',
              name: mdsLabel
            });
          }
        }
        return {
          ...obj,
          id: newId,
          ...(references && {
            references
          })
        };
      });
      await context.core.savedObjects.client.bulkCreate(updatedObjects);
      return response.ok({
        body: {
          message: 'Objects loaded successfully'
        }
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: 'Issue in loading objects for mdsId:'
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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