"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateTemplate = exports.validateInstance = void 0;
var _ajv = _interopRequireDefault(require("ajv"));
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const ajv = new _ajv.default();
const staticAsset = {
  type: 'object',
  properties: {
    path: {
      type: 'string'
    },
    annotation: {
      type: 'string',
      nullable: true
    },
    data: {
      type: 'string',
      nullable: true
    }
  },
  required: ['path'],
  additionalProperties: false
};
const templateSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    version: {
      type: 'string'
    },
    displayName: {
      type: 'string',
      nullable: true
    },
    license: {
      type: 'string'
    },
    type: {
      type: 'string'
    },
    labels: {
      type: 'array',
      items: {
        type: 'string'
      },
      nullable: true
    },
    tags: {
      type: 'array',
      items: {
        type: 'string'
      },
      nullable: true
    },
    author: {
      type: 'string',
      nullable: true
    },
    description: {
      type: 'string',
      nullable: true
    },
    sourceUrl: {
      type: 'string',
      nullable: true
    },
    workflows: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          label: {
            type: 'string'
          },
          description: {
            type: 'string'
          },
          enabled_by_default: {
            type: 'boolean'
          },
          applicable_data_sources: {
            type: 'array',
            items: {
              type: 'string'
            },
            nullable: true
          }
        },
        required: ['name', 'label', 'description', 'enabled_by_default']
      },
      nullable: true
    },
    statics: {
      type: 'object',
      properties: {
        logo: {
          ...staticAsset,
          nullable: true
        },
        gallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        },
        darkModeLogo: {
          ...staticAsset,
          nullable: true
        },
        darkModeGallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        }
      },
      additionalProperties: false,
      nullable: true
    },
    components: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          version: {
            type: 'string'
          },
          data: {
            type: 'string',
            nullable: true
          }
        },
        required: ['name', 'version']
      }
    },
    assets: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          version: {
            type: 'string'
          },
          extension: {
            type: 'string'
          },
          type: {
            type: 'string'
          },
          data: {
            type: 'string',
            nullable: true
          },
          workflows: {
            type: 'array',
            items: {
              type: 'string'
            },
            nullable: true
          }
        },
        required: ['name', 'version', 'extension', 'type'],
        additionalProperties: false
      }
    },
    sampleData: {
      type: 'object',
      properties: {
        path: {
          type: 'string'
        },
        data: {
          type: 'string',
          nullable: true
        }
      },
      required: ['path'],
      additionalProperties: false,
      nullable: true
    }
  },
  required: ['name', 'version', 'license', 'type', 'components', 'assets'],
  additionalProperties: false
};
const instanceSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    templateName: {
      type: 'string'
    },
    dataSource: {
      type: 'string'
    },
    creationDate: {
      type: 'string'
    },
    status: {
      type: 'string',
      nullable: true
    },
    assets: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          assetType: {
            type: 'string'
          },
          assetId: {
            type: 'string'
          },
          isDefaultAsset: {
            type: 'boolean'
          },
          description: {
            type: 'string'
          },
          status: {
            type: 'string',
            nullable: true
          }
        },
        required: ['assetType', 'assetId', 'isDefaultAsset', 'description']
      }
    }
  },
  required: ['name', 'templateName', 'dataSource', 'creationDate', 'assets']
};
const templateValidator = ajv.compile(templateSchema);
const instanceValidator = ajv.compile(instanceSchema);

/**
 * Validates an integration template against a predefined schema using the AJV library.
 * Since AJV validators use side effects for errors,
 * this is a more conventional wrapper that simplifies calling.
 *
 * @param data The data to be validated as an IntegrationTemplate.
 * @return A Result indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */
const validateTemplate = data => {
  if (!templateValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(templateValidator.errors))
    };
  }
  // We assume an invariant that the type of an integration is connected with its component.
  if (data.components.findIndex(x => x.name === data.type) < 0) {
    return {
      ok: false,
      error: new Error(`The integration type '${data.type}' must be included as a component`)
    };
  }
  return {
    ok: true,
    value: data
  };
};

/**
 * Validates an integration instance against a predefined schema using the AJV library.
 *
 * @param data The data to be validated as an IntegrationInstance.
 * @return A Result indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */
exports.validateTemplate = validateTemplate;
const validateInstance = data => {
  if (!instanceValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(instanceValidator.errors))
    };
  }
  return {
    ok: true,
    value: data
  };
};
exports.validateInstance = validateInstance;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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