/*******************************************************************************
* Copyright 2014-2020 Intel Corporation.
*
* This software and the related documents are Intel copyrighted  materials,  and
* your use of  them is  governed by the  express license  under which  they were
* provided to you (License).  Unless the License provides otherwise, you may not
* use, modify, copy, publish, distribute,  disclose or transmit this software or
* the related documents without Intel's prior written permission.
*
* This software and the related documents  are provided as  is,  with no express
* or implied  warranties,  other  than those  that are  expressly stated  in the
* License.
*******************************************************************************/

//@HEADER
// ***************************************************
//
// HPCG: High Performance Conjugate Gradient Benchmark
//
// Contact:
// Michael A. Heroux ( maherou@sandia.gov)
// Jack Dongarra     (dongarra@eecs.utk.edu)
// Piotr Luszczek    (luszczek@eecs.utk.edu)
//
// ***************************************************
//@HEADER

/*!
 @file CGData.hpp

 HPCG data structure
 */

#ifndef CGDATA_HPP
#define CGDATA_HPP

#include "SparseMatrix.hpp"
#include "Vector.hpp"

struct CGData_STRUCT {
  Vector r; //!< pointer to residual vector
  Vector z; //!< pointer to preconditioned residual vector
  Vector p; //!< pointer to direction vector
  Vector Ap; //!< pointer to Krylov vector

  Vector r_host;
  Vector z_host;
  Vector p_host;
  Vector Ap_host;
};
typedef struct CGData_STRUCT CGData;

/*!
 Constructor for the data structure of CG vectors.

 @param[in]  A    the data structure that describes the problem matrix and its structure
 @param[out] data the data structure for CG vectors that will be allocated to get it ready for use in CG iterations
 */
inline void InitializeSparseCGData(SparseMatrix & A, CGData & data, sycl::queue & main_queue,
                                   int runRealRef) {
  local_int_t nrow = A.localNumberOfRows;
  local_int_t ncol = A.localNumberOfColumns;
  // initialize device memory for performance
  InitializeVectorDevice(data.r, nrow, main_queue);
  InitializeVectorDevice(data.z, ncol, main_queue);
  InitializeVectorDevice(data.p, ncol, main_queue);
  InitializeVectorDevice(data.Ap, nrow, main_queue);

  if (runRealRef) {
    InitializeVectorHost(data.r_host, nrow, main_queue);
    InitializeVectorHost(data.z_host, ncol, main_queue);
    InitializeVectorHost(data.p_host, ncol, main_queue);
    InitializeVectorHost(data.Ap_host, nrow, main_queue);
  }

  return;
}

/*!
 Destructor for the CG vectors data.

 @param[inout] data the CG vectors data structure whose storage is deallocated
 */
inline void DeleteCGData(CGData & data, sycl::queue & main_queue) {

  DeleteVector (data.r, main_queue);
  DeleteVector (data.z, main_queue);
  DeleteVector (data.p, main_queue);
  DeleteVector (data.Ap, main_queue);

  DeleteVector (data.r_host, main_queue);
  DeleteVector (data.z_host, main_queue);
  DeleteVector (data.p_host, main_queue);
  DeleteVector (data.Ap_host, main_queue);

  return;
}

#endif // CGDATA_HPP

